import {TestBed, waitForAsync} from '@angular/core/testing';
import {By} from '@angular/platform-browser';
import {NoopAnimationsModule} from '@angular/platform-browser/animations';
import {ActivatedRoute, RouterModule} from '@angular/router';

import {ApiModule} from '../../lib/api/module';
import {newClient} from '../../lib/models/model_test_util';
import {ClientPageGlobalStore} from '../../store/client_page_global_store';
import {
  injectMockStore,
  STORE_PROVIDERS,
} from '../../store/store_test_providers';
import {getActivatedChildRoute, initTestEnvironment} from '../../testing';
import {CLIENT_ROUTES} from '../app/routing';
import {ApprovalCard} from '../approval_card/approval_card';
import {ClientDetailsModule} from '../client_details/module';

import {ClientPageModule} from './client_page_module';
import {FlowSection} from './flow_section';

initTestEnvironment();

describe('FlowSection', () => {
  beforeEach(waitForAsync(() => {
    TestBed.configureTestingModule({
      imports: [
        ApiModule,
        NoopAnimationsModule,
        ClientPageModule,
        ClientDetailsModule,
        RouterModule.forRoot(CLIENT_ROUTES),
      ],
      providers: [
        ...STORE_PROVIDERS,
        {provide: ActivatedRoute, useFactory: getActivatedChildRoute},
      ],
      teardown: {destroyAfterEach: false},
    }).compileComponents();
  }));

  it('shows approval if approvalsEnabled$', () => {
    const fixture = TestBed.createComponent(FlowSection);
    fixture.detectChanges();

    injectMockStore(
      ClientPageGlobalStore,
    ).mockedObservables.approvalsEnabled$.next(true);
    fixture.detectChanges();

    expect(
      fixture.debugElement.query(By.directive(ApprovalCard)),
    ).not.toBeNull();
  });

  it('does not show approval if approvalsEnabled$ is false', () => {
    const fixture = TestBed.createComponent(FlowSection);
    fixture.detectChanges();

    injectMockStore(
      ClientPageGlobalStore,
    ).mockedObservables.approvalsEnabled$.next(false);
    fixture.detectChanges();

    expect(fixture.debugElement.query(By.directive(ApprovalCard))).toBeNull();
  });

  it('sends request approval when child approval component emits the info', () => {
    const fixture = TestBed.createComponent(FlowSection);
    fixture.detectChanges();
    const client = newClient({
      clientId: 'C.1234',
      ...{},
    });

    const clientPageGlobalStore = injectMockStore(ClientPageGlobalStore);
    clientPageGlobalStore.mockedObservables.approvalsEnabled$.next(true);
    clientPageGlobalStore.mockedObservables.selectedClient$.next(client);
    fixture.detectChanges();

    fixture.debugElement
      .query(By.directive(ApprovalCard))
      .triggerEventHandler('approvalParams', {
        approvers: ['rick', 'jerry'],
        reason: 'sample reason',
        cc: [],
      });
    fixture.detectChanges();

    expect(clientPageGlobalStore.requestClientApproval).toHaveBeenCalledWith({
      clientId: 'C.1234',
      approvers: ['rick', 'jerry'],
      reason: 'sample reason',
      cc: [],
      expirationTimeUs: undefined,
    });
  });

  it('sends request approval when child approval component emits the info, with non-default duration', () => {
    const fixture = TestBed.createComponent(FlowSection);
    fixture.detectChanges();
    const client = newClient({
      clientId: 'C.1234',
      ...{},
    });

    const clientPageGlobalStore = injectMockStore(ClientPageGlobalStore);
    clientPageGlobalStore.mockedObservables.approvalsEnabled$.next(true);
    clientPageGlobalStore.mockedObservables.selectedClient$.next(client);
    fixture.detectChanges();

    fixture.debugElement
      .query(By.directive(ApprovalCard))
      .triggerEventHandler('approvalParams', {
        approvers: ['rick', 'jerry'],
        reason: 'sample reason',
        cc: [],
        expirationTimeUs: '400',
      });
    fixture.detectChanges();

    expect(clientPageGlobalStore.requestClientApproval).toHaveBeenCalledWith({
      clientId: 'C.1234',
      approvers: ['rick', 'jerry'],
      reason: 'sample reason',
      cc: [],
      expirationTimeUs: '400',
    });
  });
});
